<?php

/**
 * AJAX functions for Calendar News block
 */

// AJAX handler for loading calendar days with posts
function pyrene_ajax_load_calendar()
{
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'pyrene_calendar_news_nonce')) {
        wp_send_json_error(['message' => 'Invalid nonce']);
    }

    // Get parameters
    $month = isset($_POST['month']) ? intval($_POST['month']) : date('n');
    $year = isset($_POST['year']) ? intval($_POST['year']) : date('Y');
    $category_id = isset($_POST['category_id']) ? intval($_POST['category_id']) : 0;

    // Get days in month
    $days_in_month = cal_days_in_month(CAL_GREGORIAN, $month, $year);


    $start_date = $year . '-' . sprintf('%02d', $month) . '-01';
    $end_date = $year . '-' . sprintf('%02d', $month) . '-' . $days_in_month;

    // Get days with posts
    $days_with_posts = [];
    if ($category_id > 0) {
        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'cat' => $category_id,
            'meta_query' => array(
                array(
                    'key' => '_event_date',
                    'value' => array($start_date, $end_date),
                    'compare' => 'BETWEEN',
                    'type' => 'DATE'
                )
            ),
            'posts_per_page' => -1,
        );
        $query = new WP_Query($args);

        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $event_date = carbon_get_post_meta(get_the_ID(), 'event_date');
                if (!empty($event_date)) {
                    $event_day = intval(date('j', strtotime($event_date)));
                    if (!in_array($event_day, $days_with_posts)) {
                        $days_with_posts[] = $event_day;
                    }
                }
            }
        }
        wp_reset_postdata();
    }

    // Sort days with posts
    sort($days_with_posts);

    // Send response
    wp_send_json_success([
        'days_in_month' => $days_in_month,
        'days_with_posts' => $days_with_posts,
        'month' => $month,
        'year' => $year,
    ]);
}
add_action('wp_ajax_pyrene_load_calendar', 'pyrene_ajax_load_calendar');
add_action('wp_ajax_nopriv_pyrene_load_calendar', 'pyrene_ajax_load_calendar');

// AJAX handler for loading post for a specific date
function pyrene_ajax_load_post_for_date()
{
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'pyrene_calendar_news_nonce')) {
        wp_send_json_error(['message' => 'Invalid nonce']);
    }

    // Get parameters
    $day = isset($_POST['day']) ? intval($_POST['day']) : date('j');
    $month = isset($_POST['month']) ? intval($_POST['month']) : date('n');
    $year = isset($_POST['year']) ? intval($_POST['year']) : date('Y');
    $category_id = isset($_POST['category_id']) ? intval($_POST['category_id']) : 0;

    // Get post for date
    $post_data = null;
    $date_formatted = $year . '-' . sprintf('%02d', $month) . '-' . sprintf('%02d', $day);

    if ($category_id > 0) {
        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'cat' => $category_id,
            'meta_query' => array(
                array(
                    'key' => '_event_date',
                    'value' => $date_formatted,
                    'compare' => '=',
                    'type' => 'DATE'
                )
            ),
            'posts_per_page' => 1,
        );
        $query = new WP_Query($args);

        if ($query->have_posts()) {
            $query->the_post();

            // Get featured image
            $image_url = '';
            if (has_post_thumbnail()) {
                $image_url = get_the_post_thumbnail_url(null, 'large');
            }

            // Get image credit
            $image_credit = '';
            $image_id = get_post_thumbnail_id();
            if ($image_id) {
                $img_credit = get_post_meta($image_id, '_credit', true);
                if (!empty($img_credit)) {
                    $image_credit = $img_credit;
                } else {
                    // Fallback to caption
                    $image_post = get_post($image_id);
                    if ($image_post && !empty($image_post->post_excerpt)) {
                        $image_credit = $image_post->post_excerpt;
                    }
                }
            }

            // Get category
            $category_name = '';
            $categories = get_the_category();
            if (!empty($categories)) {
                $category_name = $categories[0]->name;
            }

            // Get excerpt
            $excerpt = has_excerpt() ? get_the_excerpt() : wp_trim_words(get_the_content(), 30, '...');

            // Create post data
            $post_data = array(
                'id' => get_the_ID(),
                'title' => get_the_title(),
                'excerpt' => wpautop($excerpt),
                'permalink' => get_permalink(),
                'image' => $image_url,
                'image_credit' => $image_credit,
                'category' => $category_name,
                'date' => get_the_date('j F Y'),
            );
        }
        wp_reset_postdata();
    }

    // Send response
    wp_send_json_success([
        'post' => $post_data,
        'day' => $day,
        'month' => $month,
        'year' => $year,
    ]);
}
add_action('wp_ajax_pyrene_load_post_for_date', 'pyrene_ajax_load_post_for_date');
add_action('wp_ajax_nopriv_pyrene_load_post_for_date', 'pyrene_ajax_load_post_for_date');
