<?php

/**
 * Filtre simplifié pour contenu premium avec préservation de balisage basique
 */

// Filtrer le contenu pour les articles premium
add_filter('the_content', 'filter_premium_content', 20);

function filter_premium_content($content)
{
    // Ne s'applique qu'aux articles individuels
    if (!is_singular('post')) {
        return $content;
    }

    // Vérifier si l'article est premium
    $is_premium = carbon_get_post_meta(get_the_ID(), 'is_premium_content');

    // Si non premium, retourner le contenu normal
    if (!$is_premium) {
        return $content;
    }

    // Vérifier si l'utilisateur est connecté
    if (is_user_logged_in_by_cookie()) {
        return $content;
    }

    // Extraire une partie du contenu en préservant certaines balises
    $preview_content = get_limited_html_content($content, 200);


    ob_start();
?>
    <div class="premium-content-preview">
        <?php echo $preview_content; ?>
        <div class="content-fade"></div>
    </div>

    <div class="premium-content-cta">
        <div class="subscription-message">
            <p>POUR LIRE LA SUITE, ABONNEZ-VOUS</p>
        </div>

        <div class="subscription-options">
            <div class="subscription-options__choices">
                <div class="option">
                    <div class="option__container">
                        <span class="duration">15 jours</span>
                        <span class="price">Pour 1€</span>
                    </div>
                    <a target="_blank" href="https://magweb.net-ful.com/site/pyreneesmagazine/abonnements-pyrenees-magazine/fr/kiosque/abonnement.html" class="subscribe-button">ABONNEZ-VOUS !</a>
                </div>

                <div class="option">
                    <div class="option__container">
                        <span class="duration">1 an</span>
                        <span class="price">Pour 60€</span>
                    </div>
                    <a target="_blank" href="https://magweb.net-ful.com/site/pyreneesmagazine/abonnements-pyrenees-magazine/fr/kiosque/abonnement.html" class="subscribe-button">ABONNEZ-VOUS !</a>
                </div>
            </div>

            <div class="option option--login">
                <a target="_blank" href="https://magweb.net-ful.com/site/pyreneesmagazine/default/fr/kiosque/compte-identification.html" class="subscribe-button">SE CONNECTER</a>
            </div>
        </div>
    </div>



<?php
    return ob_get_clean();
}

/**
 * Extrait un contenu HTML limité en préservant le balisage essentiel
 */
function get_limited_html_content($content, $max_words)
{
    // Supprimer les scripts et styles
    $content = preg_replace('/<script\b[^>]*>(.*?)<\/script>/is', '', $content);
    $content = preg_replace('/<style\b[^>]*>(.*?)<\/style>/is', '', $content);

    // DOM pour manipuler le contenu
    $dom = new DOMDocument();
    libxml_use_internal_errors(true);
    @$dom->loadHTML(mb_convert_encoding('<div>' . $content . '</div>', 'HTML-ENTITIES', 'UTF-8'));
    libxml_clear_errors();

    $xpath = new DOMXPath($dom);
    $body = $dom->getElementsByTagName('body')->item(0);
    $container = $body->getElementsByTagName('div')->item(0);

    // Compter les mots dans chaque nœud texte
    $word_count = 0;
    $preview_html = '';
    $reached_limit = false;

    // Parcourir les nœuds de premier niveau
    foreach ($container->childNodes as $child) {
        if ($reached_limit) break;

        // Cloner le nœud pour le manipuler
        $node = $child->cloneNode(true);

        // Traiter différemment selon le type de nœud
        if ($node->nodeType === XML_TEXT_NODE) {
            $text = trim($node->nodeValue);
            $words = preg_split('/\s+/', $text);
            $word_count_in_node = count($words);

            if ($word_count + $word_count_in_node > $max_words) {
                $remaining = $max_words - $word_count;
                if ($remaining > 0) {
                    $preview_html .= implode(' ', array_slice($words, 0, $remaining)) . '...';
                }
                $reached_limit = true;
            } else {
                $preview_html .= $text;
                $word_count += $word_count_in_node;
            }
        } elseif ($node->nodeType === XML_ELEMENT_NODE) {
            // Si c'est un paragraphe, titre, liste, etc.
            $inner_text = $node->textContent;
            $inner_words = preg_split('/\s+/', $inner_text);
            $inner_word_count = count($inner_words);

            if ($word_count + $inner_word_count <= $max_words) {
                // Si tout le contenu du nœud rentre, l'ajouter tel quel
                $preview_html .= $dom->saveHTML($node);
                $word_count += $inner_word_count;
            } else {
                // Sinon, créer une version tronquée du nœud
                $truncated_node = truncate_node($node, $max_words - $word_count);
                if ($truncated_node) {
                    $preview_html .= $dom->saveHTML($truncated_node);
                }
                $reached_limit = true;
            }
        }
    }

    return $preview_html;
}

/**
 * Tronque un nœud DOM pour ne conserver que le nombre de mots spécifié
 */
function truncate_node($node, $max_words)
{
    // Si c'est un nœud texte, le tronquer directement
    if ($node->nodeType === XML_TEXT_NODE) {
        $text = trim($node->nodeValue);
        $words = preg_split('/\s+/', $text);
        if (count($words) > $max_words) {
            $node->nodeValue = implode(' ', array_slice($words, 0, $max_words)) . '...';
        }
        return $node;
    }

    // Si c'est un élément, parcourir ses enfants
    if ($node->nodeType === XML_ELEMENT_NODE) {
        $word_count = 0;
        $nodes_to_remove = [];

        // Parcourir les nœuds enfants
        foreach ($node->childNodes as $child) {
            if ($word_count >= $max_words) {
                $nodes_to_remove[] = $child;
                continue;
            }

            $child_text = $child->textContent;
            $child_words = preg_split('/\s+/', $child_text);
            $child_word_count = count($child_words);

            if ($word_count + $child_word_count > $max_words) {
                // Tronquer ce nœud enfant
                $remaining = $max_words - $word_count;
                if ($remaining > 0) {
                    truncate_node($child, $remaining);
                    $word_count = $max_words;
                } else {
                    $nodes_to_remove[] = $child;
                }
            } else {
                // Garder ce nœud enfant intact
                $word_count += $child_word_count;
            }
        }

        // Supprimer les nœuds en trop
        foreach ($nodes_to_remove as $child) {
            if ($child->parentNode) {
                $child->parentNode->removeChild($child);
            }
        }

        return $node;
    }

    return null;
}
