<?php
/**
 * Fonctions d'authentification pour gérer le cookie Magellan
 */

/**
 * Vérifie si l'utilisateur est connecté en analysant le cookie Magellan
 * 
 * @param string $encryption_key La clé de chiffrement fournie par le prestataire
 * @return array|false Informations de l'utilisateur ou false si non connecté
 */
function is_user_logged_in_via_cookie($encryption_key) {
    // Nom du cookie à vérifier
    $cookie_name = 'magellan';
    
    // Vérifier si le cookie existe
    if (!isset($_COOKIE[$cookie_name])) {
        return false;
    }
    
    // Récupérer la valeur du cookie
    $encrypted_data = $_COOKIE[$cookie_name];
    
    try {
        // Déchiffrer le contenu du cookie
        $decrypted_data = decrypt_cookie_data($encrypted_data, $encryption_key);
        
        // Si le format est JSON (nouveau format avec infos d'abonnement)
        if (is_valid_json($decrypted_data)) {
            $user_data = json_decode($decrypted_data, true);
            return $user_data;
        } 
        // Sinon, c'est l'ancien format (numéro de donateur|date)
        else {
            // Diviser la chaîne en numéro de donateur et date
            $parts = explode('|', $decrypted_data);
            
            if (count($parts) !== 2) {
                return false; // Format invalide
            }
            
            $donor_number = $parts[0];
            $validity_date = $parts[1];
            
            // Vérifier si la date de validité est dépassée
            $current_date = new DateTime();
            $validity = DateTime::createFromFormat('d/m/Y H:i:s', $validity_date);
            
            if ($validity === false || $current_date > $validity) {
                return false; // Date de validité dépassée
            }
            
            // Créer un tableau avec les informations basiques de l'utilisateur
            $user_data = [
                'contact_number' => $donor_number,
                'titles' => []
            ];
            
            return $user_data;
        }
    } catch (Exception $e) {
        // En cas d'erreur de déchiffrement
        error_log('Erreur de déchiffrement du cookie : ' . $e->getMessage());
        return false;
    }
}

/**
 * Déchiffre les données du cookie avec AES-256-CBC
 * 
 * @param string $ciphertext Texte chiffré
 * @param string $key Clé de chiffrement (en hexadécimal)
 * @return string Texte déchiffré
 */
function decrypt_cookie_data($hexCookie, $hexKey) {
    if (!ctype_xdigit($hexKey)) {
        return false;
    }
    
    $binKey = hex2bin($hexKey);
    $binaryCookie = hex2bin($hexCookie);
    
    // Extraire l'IV (16 premiers octets)
    $iv = substr($binaryCookie, 0, 16);
    
    // Extraire le contenu chiffré (le reste)
    $ciphertext = substr($binaryCookie, 16);
    
    // Déchiffrer avec AES-128-CBC
    $plaintext = openssl_decrypt(
        $ciphertext, 
        'aes-128-cbc', 
        $binKey, 
        OPENSSL_RAW_DATA, 
        $iv
    );
    
    // Retourner le résultat déchiffré
    return $plaintext;
}

/**
 * Vérifie si une chaîne est un JSON valide
 * 
 * @param string $string Chaîne à vérifier
 * @return bool True si c'est du JSON valide, false sinon
 */
function is_valid_json($string) {
    json_decode($string);
    return (json_last_error() == JSON_ERROR_NONE);
}

/**
 * Génère l'URL de redirection vers le site externe d'authentification
 * 
 * @param string $external_auth_url URL du site externe d'authentification
 * @param string $redirect_url URL de redirection après authentification
 * @return string URL complète
 */
function get_external_auth_url($external_auth_url, $redirect_url = '/') {
    // Encoder l'URL de redirection pour une utilisation sécurisée dans l'URL
    $encoded_redirect = urlencode($redirect_url);
    
    // Construire l'URL complète
    $auth_url = $external_auth_url;
    
    // Ajouter le paramètre de redirection
    if (strpos($auth_url, '?') !== false) {
        $auth_url .= '&redirect_url=' . $encoded_redirect;
    } else {
        $auth_url .= '?redirect_url=' . $encoded_redirect;
    }
    
    return $auth_url;
}

/**
 * Vérifie si un utilisateur a un abonnement actif pour un titre spécifique
 * 
 * @param array $user_data Données de l'utilisateur
 * @param string $title_code Code du titre à vérifier
 * @return bool True si l'utilisateur a un abonnement actif, false sinon
 */
function has_active_subscription($user_data, $title_code) {
    // Si les données utilisateur sont invalides ou ne contiennent pas de titres
    if (!$user_data || !isset($user_data['titles']) || !is_array($user_data['titles'])) {
        return false;
    }
    
    // Parcourir les titres pour trouver celui qui correspond
    foreach ($user_data['titles'] as $title) {
        if (isset($title['title_code']) && $title['title_code'] === $title_code) {
            // Vérifier si l'abonnement est actif et non suspendu
            if (isset($title['is_current']) && $title['is_current'] === true && 
                isset($title['is_suspend']) && $title['is_suspend'] === false) {
                return true;
            }
        }
    }
    
    return false;
}

/**
 * Récupère tous les abonnements actifs d'un utilisateur
 * 
 * @param array $user_data Données de l'utilisateur
 * @return array Liste des codes de titres actifs
 */
function get_active_subscriptions($user_data) {
    $active_titles = [];
    
    // Si les données utilisateur sont invalides ou ne contiennent pas de titres
    if (!$user_data || !isset($user_data['titles']) || !is_array($user_data['titles'])) {
        return $active_titles;
    }
    
    // Parcourir les titres et ajouter ceux qui sont actifs
    foreach ($user_data['titles'] as $title) {
        if (isset($title['title_code']) && 
            isset($title['is_current']) && $title['is_current'] === true && 
            isset($title['is_suspend']) && $title['is_suspend'] === false) {
            $active_titles[] = $title['title_code'];
        }
    }
    
    return $active_titles;
}

/**
 * Récupère le numéro de contact de l'utilisateur connecté
 * 
 * @param array $user_data Données de l'utilisateur
 * @return string|null Numéro de contact ou null si non disponible
 */
function get_contact_number($user_data) {
    return isset($user_data['contact_number']) ? $user_data['contact_number'] : null;
}

/**
 * Récupère l'email/login de l'utilisateur connecté (nouveau format uniquement)
 * 
 * @param array $user_data Données de l'utilisateur
 * @return string|null Email/login de l'utilisateur ou null si non disponible
 */
function get_user_login($user_data) {
    return isset($user_data['login']) ? $user_data['login'] : null;
}