document.addEventListener('DOMContentLoaded', function () {
    // Select all calendar news blocks on the page
    const calendarBlocks = document.querySelectorAll('.calendar-news');

    calendarBlocks.forEach(function (block) {
        const blockId = block.id;
        const prevButton = block.querySelector('.calendar-news__nav-prev');
        const nextButton = block.querySelector('.calendar-news__nav-next');
        const monthTitle = block.querySelector('.calendar-news__month-title');
        let contentContainer;

        // Vérifier si l'élément existe directement dans le bloc
        contentContainer = block.querySelector('.calendar-news__content');

        // S'il n'existe pas, essayer de le trouver par ID
        if (!contentContainer) {
            const contentId = `calendar-news-content-${blockId.replace('calendar-news-', '')}`;
            contentContainer = document.getElementById(contentId);

            // Si toujours null, utiliser une approche alternative
            if (!contentContainer) {
                // Chercher le conteneur de contenu dans le bloc parent
                contentContainer = block.querySelector('[id^="calendar-news-content"]');
            }
        }

        // Si toujours null, afficher un message d'erreur dans la console
        if (!contentContainer) {
            console.error('Conteneur de contenu non trouvé:', blockId);
        }

        let currentMonth = parseInt(block.dataset.month, 10);
        let currentYear = parseInt(block.dataset.year, 10);
        const today = new Date();
        const todayDay = today.getDate();
        const todayMonth = today.getMonth() + 1; // JavaScript months are 0-based
        const todayYear = today.getFullYear();

        // Month names in French
        const monthNames = [
            'JANVIER', 'FÉVRIER', 'MARS', 'AVRIL', 'MAI', 'JUIN',
            'JUILLET', 'AOÛT', 'SEPTEMBRE', 'OCTOBRE', 'NOVEMBRE', 'DÉCEMBRE'
        ];

        // Add event listeners for navigation buttons
        if (prevButton && nextButton) {
            prevButton.addEventListener('click', function () {
                navigateMonth(-1);
            });

            nextButton.addEventListener('click', function () {
                navigateMonth(1);
            });
        }

        // Add event listeners for days with posts
        const dayButtons = block.querySelectorAll('.calendar-news__day--has-post');
        dayButtons.forEach(function (button) {
            button.addEventListener('click', function () {
                const day = parseInt(button.dataset.day, 10);
                loadPostForDate(day, currentMonth, currentYear);

                // Update active state
                dayButtons.forEach(btn => btn.classList.remove('calendar-news__day--active'));
                button.classList.add('calendar-news__day--active');
            });
        });

        // Automatically select today if it has posts
        const todayButton = block.querySelector(`.calendar-news__day--today.calendar-news__day--has-post`);
        if (todayButton) {
            todayButton.classList.add('calendar-news__day--active');
        }

        // Function to navigate between months
        function navigateMonth(direction) {
            let newMonth = currentMonth + direction;
            let newYear = currentYear;

            if (newMonth > 12) {
                newMonth = 1;
                newYear++;
            } else if (newMonth < 1) {
                newMonth = 12;
                newYear--;
            }

            // Update current month and year
            currentMonth = newMonth;
            currentYear = newYear;

            // Update month title
            monthTitle.textContent = monthNames[currentMonth - 1];

            // Load calendar for new month
            loadCalendarForMonth(currentMonth, currentYear);
        }

        // Function to load calendar for a specific month
        function loadCalendarForMonth(month, year) {
            // Show loading state
            const daysContainer = block.querySelector('.calendar-news__days');
            daysContainer.innerHTML = '<div class="calendar-news__loading">Chargement...</div>';

            // Make AJAX request to get days with posts for the month
            const xhr = new XMLHttpRequest();
            xhr.open('POST', pyreneCalendarData.ajaxurl, true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

            xhr.onload = function () {
                if (xhr.status === 200) {
                    try {
                        const response = JSON.parse(xhr.responseText);

                        if (response.success) {
                            // Render calendar days
                            renderCalendarDays(response.data.days_in_month, response.data.days_with_posts, month, year);

                            // Load post for first day with post or clear content if none
                            if (response.data.days_with_posts.length > 0) {
                                // If today is in this month and has posts, select it
                                if (month === todayMonth && year === todayYear && response.data.days_with_posts.includes(todayDay)) {
                                    loadPostForDate(todayDay, month, year);
                                } else {
                                    // Otherwise select first day with posts
                                    loadPostForDate(response.data.days_with_posts[0], month, year);
                                }
                            } else {
                                // No posts this month
                                contentContainer.innerHTML = `
                                    <div class="calendar-news__no-content">
                                        <p>Aucune actualité pour ce mois.</p>
                                    </div>
                                `;
                            }
                        } else {
                            console.error('Error loading calendar:', response.data.message);
                        }
                    } catch (e) {
                        console.error('Error parsing response:', e);
                    }
                }
            };

            xhr.onerror = function () {
                console.error('Request failed');
            };

            const data = `action=pyrene_load_calendar&nonce=${pyreneCalendarData.nonce}&month=${month}&year=${year}&category_id=${pyreneCalendarData.category_id}`;
            xhr.send(data);
        }

        // Function to render calendar days
        function renderCalendarDays(daysInMonth, daysWithPosts, month, year) {
            const daysContainer = block.querySelector('.calendar-news__days');
            let daysHTML = '';

            for (let day = 1; day <= daysInMonth; day++) {
                const hasPost = daysWithPosts.includes(day);
                const isToday = (day === todayDay && month === todayMonth && year === todayYear);

                let dayClasses = 'calendar-news__day';
                if (hasPost) {
                    dayClasses += ' calendar-news__day--has-post';
                }
                if (isToday) {
                    dayClasses += ' calendar-news__day--today';
                }

                const ariaLabel = `${day} ${monthNames[month - 1]}` +
                    (hasPost ? ' - Contient des actualités' : '') +
                    (isToday ? ' - Aujourd\'hui' : '');

                if (hasPost) {
                    daysHTML += `
                        <button 
                            type="button"
                            class="${dayClasses}"
                            data-day="${day}"
                            aria-label="${ariaLabel}"
                            ${isToday ? 'aria-current="date"' : ''}
                        >
                            ${day}
                        </button>
                    `;
                } else {
                    daysHTML += `
                        <span 
                            class="${dayClasses}"
                            aria-label="${ariaLabel}"
                            ${isToday ? 'aria-current="date"' : ''}
                        >
                            ${day}
                        </span>
                    `;
                }
            }

            daysContainer.innerHTML = daysHTML;

            // Add event listeners for the new day buttons
            const newDayButtons = block.querySelectorAll('.calendar-news__day--has-post');
            newDayButtons.forEach(function (button) {
                button.addEventListener('click', function () {
                    const day = parseInt(button.dataset.day, 10);
                    loadPostForDate(day, month, year);

                    // Update active state
                    newDayButtons.forEach(btn => btn.classList.remove('calendar-news__day--active'));
                    button.classList.add('calendar-news__day--active');
                });
            });

            // Auto-select today or first day with post
            let dayToSelect = null;

            if (month === todayMonth && year === todayYear && daysWithPosts.includes(todayDay)) {
                dayToSelect = block.querySelector(`.calendar-news__day--today.calendar-news__day--has-post`);
            }

            if (!dayToSelect && daysWithPosts.length > 0) {
                dayToSelect = block.querySelector(`.calendar-news__day--has-post[data-day="${daysWithPosts[0]}"]`);
            }

            if (dayToSelect) {
                dayToSelect.classList.add('calendar-news__day--active');
            }
        }

        // Function to load post for a specific date
        function loadPostForDate(day, month, year) {
            // Show loading state
            contentContainer.innerHTML = '<div class="calendar-news__loading">Chargement...</div>';

            // Make AJAX request to get post for the date
            const xhr = new XMLHttpRequest();
            xhr.open('POST', pyreneCalendarData.ajaxurl, true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

            xhr.onload = function () {
                if (xhr.status === 200) {
                    try {
                        const response = JSON.parse(xhr.responseText);

                        if (response.success && response.data.post) {
                            const post = response.data.post;

                            // Create HTML for post
                            let postHTML = `
                                <article class="calendar-news__article">
                                    <div class="calendar-news__image-container">
                                        ${post.image ? `<img src="${post.image}" alt="${post.title}" class="calendar-news__image">` : ''}
                                    </div>
                                    
                                    <div class="calendar-news__article-content">
                                        ${post.category ? `
                                            <div class="calendar-news__category">
                                                <span class="calendar-news__category-label">${post.category}</span>
                                            </div>
                                        ` : ''}
                                        
                                        <h3 class="calendar-news__article-title">
                                            ${post.title}
                                        </h3>
                                        
                                        <div class="calendar-news__article-excerpt">
                                            ${post.excerpt}
                                        </div>
                                        
                                        <div class="calendar-news__article-action">
                                            <a href="${post.permalink}" class="calendar-news__button">
                                                ${pyreneCalendarData.read_more_text}
                                            </a>
                                        </div>
                                    </div>
                                </article>
                            `;

                            if (post.image_credit) {
                                postHTML += `
                                    <div class="calendar-news__image-credit">
                                        ${post.image_credit}
                                    </div>
                                `;
                            }

                            contentContainer.innerHTML = postHTML;

                        } else {
                            // No post for this date
                            contentContainer.innerHTML = `
                                <div class="calendar-news__no-content">
                                    <p>Aucune actualité pour cette date.</p>
                                </div>
                            `;
                        }
                    } catch (e) {
                        console.error('Error parsing response:', e);
                    }
                }
            };

            xhr.onerror = function () {
                console.error('Request failed');
            };

            const data = `action=pyrene_load_post_for_date&nonce=${pyreneCalendarData.nonce}&day=${day}&month=${month}&year=${year}&category_id=${pyreneCalendarData.category_id}`;
            xhr.send(data);
        }
    });
});